/**
* \file: daemon_model.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <unistd.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>

#include "model/daemon_model.h"
#include "encryption/sdc_access.h"

#include "control/configuration.h"
#include "util/logger.h"

#include <systemd/sd-daemon.h>

/*
 * It might happen that a level change couldn't be performed completely (without errors).
 * Anyway it makes no sense to recovery the previous level or this recovery failed itself.
 * In that case we will use this value instead of the target vale.
 * Note: This value is only used for target levels.
 * Note: No further level change is triggered if this value is being read
 * Note: The value is only used inside the daemon model and the ALD state file.
 *       When the target value is read by other parts of the ALD the value is
 *       mapped to the actual level value (current level value).
 *
 * Value needs to be larger than MAX_LEVEL
 */
#define LEVEL_CHANGE_COMPLETED_WITH_ERRORS	    0xFFFFFFF0 /* used to mark an completed level change with errors */

/*
 * This special value is used to trigger recovery to fail-safe-level without overwriting current level
 * Note: This value is only used for target levels.
 * Note: The value is only used inside the daemon model and the ALD state file.
 * 		 When the target value is read by other parts of the ALD the value is
 * 		 mapped to the actual level value (current level value).
 *
 * Value needs to be larger than MAX_LEVEL
 */
#define LEVEL_RECOVERY_TO_FAIL_SAFE                0xFFFFFFFD /* used to mark recovery to fail-safe-level needed */

/*
 * In order to distinguish a level change from level X to X from
 * a stable level X (current and target value in the state file are identical) we
 * use this special level value.
 * Note: This value is only used for target levels.
 * Note: The value is only used inside the daemon model and the ALD state file.
 * 		 When the target value is read by other parts of the ALD the value is
 * 		 mapped to the actual level value (current level value).
 *
 * Value needs to be larger than MAX_LEVEL
 */
#define TARGET_LEVEL_IS_SAME_AS_CURRENT             0xFFFFFFFE /* used to mark an ongoing level change to the same as the current level */

/*
 * In case the levels are completely invalid we want to ensure that
 * we recover to the fail-safe level in the next power cycle.
 *
 * Value needs to be larger than MAX_LEVEL
 */
#define LEVEL_IS_INVALID                            0xFFFFFFFF /* used to mark an invalid level - needing recovery */

//-------------------------------------- private attributes ----------------------------------------------------------
static daemon_model_t daemon_model=
{
		DAEMON_LEVEL_INITIALIZATION,
		DAEMON_SERVICE_INITIALIZATION,
		0,
		0,
		{0},
		{0}
};

static const char *daemon_level_state_names[]=
{
		[DAEMON_LEVEL_INITIALIZATION] = "DAEMON_LEVEL_INITIALIZATION",
		[DAEMON_LEVEL_STABLE] = "DAEMON_LEVEL_STABLE",
		[DAEMON_LEVEL_RECOVERY_ONGOING] = "DAEMON_LEVEL_RECOVERY_ONGOING",
		[DAEMON_LEVEL_CHANGE_ONGOING] = "DAEMON_LEVEL_CHANGE_ONGOING",
		[DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED] = "DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED"
};
static const char *daemon_service_state_names[]=
{
		[DAEMON_SERVICE_INITIALIZATION] = "DAEMON_SERVICE_INITIALIZATION",
		[DAEMON_SERVICE_WAITING_FOR_APP_IFACE_READY] = "DAEMON_SERVICE_WAITING_FOR_APP_IFACE_READY",
		[DAEMON_SERVICE_WAITING_FOR_LEVEL_CHANGE_RECOVERY_ON_STARTUP] = "DAEMON_SERVICE_WAITING_FOR_LEVEL_CHANGE_RECOVERY_ON_STARTUP",
		[DAEMON_SERVICE_READY] = "DAEMON_SERVICE_READY",
		[DAEMON_SERVICE_SHUTDOWN_PENDING] = "DAEMON_SERVICE_SHUTDOWN_PENDING",
};
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- private function declaration ------------------------------------------------
static error_code_t daemon_model_load_persistent_model(persistent_model_t *model);

static error_code_t daemon_model_store_persistent_model(persistent_model_t *model);

static void daemon_model_set_service_state(daemon_service_state_t new_state);

//--------------------------------------------------------------------------------------------------------------------


//-------------------------------------- init / deinit / persist function definitions --------------------------------
error_code_t daemon_model_init(void)
{
	error_code_t result;

	daemon_model.daemon_level_state = DAEMON_LEVEL_INITIALIZATION;
	daemon_model.daemon_service_state = DAEMON_SERVICE_INITIALIZATION;
	daemon_model.persistent_model.current_persisted_level = FAIL_SAFE_LEVEL;
	daemon_model.persistent_model.targeted_persisted_level = LEVEL_IS_INVALID;

	result=daemon_model_load_persistent_model(&(daemon_model.state_file_model));

	/*
	 * Check validity of ALD state file settings
	 * - current level must be a valid level
	 * - target level must be a valid level
	 */
	if ((result == RESULT_OK) &&
		((daemon_model.state_file_model.current_persisted_level > MAX_LEVEL) ||
		 ((daemon_model.state_file_model.targeted_persisted_level > MAX_LEVEL) &&
		  (daemon_model.state_file_model.targeted_persisted_level != TARGET_LEVEL_IS_SAME_AS_CURRENT) &&
		  (daemon_model.state_file_model.targeted_persisted_level != LEVEL_RECOVERY_TO_FAIL_SAFE) &&
		  (daemon_model.state_file_model.targeted_persisted_level != LEVEL_CHANGE_COMPLETED_WITH_ERRORS))))
	{
			/*
			 * don't dump invalid levels -- this might be used by an attacker to decrypt
			 * 8 bytes of data with ALD key
			 */
			logger_log_error("Invalid levels loaded from ALD state file.");
			logger_log_errmem("Invalid levels loaded from ALD state file.");

			result = RESULT_INVALID_LEVEL;
	}

	if (result == RESULT_OK)
		/* use same values of state file */
		memcpy(&(daemon_model.persistent_model),
			   &(daemon_model.state_file_model),
			   sizeof(persistent_model_t));
	else
		/* use defaults */
		logger_log_info("Trigger switch to fail-safe levels.");

	daemon_model.current_active_level=daemon_model.persistent_model.current_persisted_level;
	daemon_model.targeted_active_level=daemon_model.persistent_model.targeted_persisted_level;

	return RESULT_OK;
}

error_code_t daemon_model_persist(void)
{
	error_code_t result;

	/* don't write state file if unchanged */
	if (memcmp(&(daemon_model.persistent_model),
			   &(daemon_model.state_file_model),
			   sizeof(persistent_model_t)) == 0)
		return RESULT_OK;

	result = daemon_model_store_persistent_model(&(daemon_model.persistent_model));

	if (result == RESULT_OK) {
		/* state file has been written successfully -> update struct */
		memcpy(&(daemon_model.state_file_model),
			   &(daemon_model.persistent_model),
			   sizeof(persistent_model_t));
	} else {
		/* error writing state file -> invalidate struct */
		daemon_model.state_file_model.current_persisted_level = LEVEL_IS_INVALID;
		daemon_model.state_file_model.targeted_persisted_level = LEVEL_IS_INVALID;
	}

	return result;
}

void daemon_model_deinit(void)
{
	switch (daemon_model.daemon_level_state) {
	case DAEMON_LEVEL_CHANGE_ONGOING:
	case DAEMON_LEVEL_RECOVERY_ONGOING:
		logger_log_info("WARNING: Stopping daemon while operations are ongoing - level state %s.",
				daemon_model_get_level_state_name(daemon_model.daemon_level_state));
		break;
	case DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED:
		logger_log_info("Stopping daemon in level state LEVEL_RECOVERY_ONSTARTUP_NEEDED.");
		break;
	default:
		/* nothing to log here */
		break;
	}
}
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- get functions for the levels ------------------------------------------------

static security_level_t daemon_model_to_actual_current_level(security_level_t internal_current_level)
{
	switch (internal_current_level) {
	case LEVEL_CHANGE_COMPLETED_WITH_ERRORS: /* not valid for current, but map to FAIL SAFE - just in case */
	case TARGET_LEVEL_IS_SAME_AS_CURRENT: /* not valid for current, but map to FAIL SAFE - just in case */
	case LEVEL_RECOVERY_TO_FAIL_SAFE: /* not valid for current, but map to FAIL SAFE - just in case */
	case LEVEL_IS_INVALID:
		return FAIL_SAFE_LEVEL;
	default:
		/* no special flags used */
		return internal_current_level;
	}
}

static security_level_t daemon_model_to_actual_target_level(security_level_t internal_target_level,
															security_level_t internal_current_level)
{
	switch (internal_target_level) {
	case LEVEL_CHANGE_COMPLETED_WITH_ERRORS:
	case TARGET_LEVEL_IS_SAME_AS_CURRENT:
		return daemon_model_to_actual_current_level(internal_current_level);
	case LEVEL_RECOVERY_TO_FAIL_SAFE:
	case LEVEL_IS_INVALID:
		return FAIL_SAFE_LEVEL;
	default:
		/* no special flags used */
		return internal_target_level;
	}
}

security_level_t daemon_model_get_active_level(void)
{
	return daemon_model_to_actual_current_level(daemon_model.current_active_level);
}

security_level_t daemon_model_get_targeted_active_level(void)
{
	return daemon_model_to_actual_target_level(daemon_model.targeted_active_level,
											   daemon_model.current_active_level);
}

security_level_t daemon_model_get_persisted_level(void)
{
	return daemon_model_to_actual_current_level(daemon_model.persistent_model.current_persisted_level);
}

security_level_t daemon_model_get_targeted_persisted_level(void)
{
	return daemon_model_to_actual_target_level(daemon_model.persistent_model.targeted_persisted_level,
											   daemon_model.persistent_model.current_persisted_level);
}
//--------------------------------------------------------------------------------------------------------------------

//---------------------------------- information on incomplete transactions ------------------------------------------------

bool daemon_model_is_recovery_needed(void)
{
	/* some level is invalid - trigger recovery for sure */
	if ((daemon_model.persistent_model.current_persisted_level == LEVEL_IS_INVALID) ||
		(daemon_model.persistent_model.targeted_persisted_level == LEVEL_IS_INVALID) ||
		/* recovery to fail-safe-level */
		(daemon_model.persistent_model.targeted_persisted_level == LEVEL_RECOVERY_TO_FAIL_SAFE) ||
		/* replay with timeout or reboot during an ongoing level change X to X */
		(daemon_model.persistent_model.targeted_persisted_level == TARGET_LEVEL_IS_SAME_AS_CURRENT))
		return true;

	/* previous level change wasn't completely successful - anyway we won't try again */
	if (daemon_model.persistent_model.targeted_persisted_level == LEVEL_CHANGE_COMPLETED_WITH_ERRORS) {
		logger_log_info("Warning: Current persistent level(%d) has been set with errors.",
				(int)daemon_model.persistent_model.current_persisted_level);
		return false;
	}

	return daemon_model.persistent_model.current_persisted_level!=
			daemon_model.persistent_model.targeted_persisted_level;
}

security_level_t daemon_model_get_recovery_level(void)
{
	if ((daemon_model.persistent_model.targeted_persisted_level == LEVEL_RECOVERY_TO_FAIL_SAFE))
	{
		return FAIL_SAFE_LEVEL;
	}

	return daemon_model_get_persisted_level();
}

//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- daemon state functions ------------------------------------------------------
static void daemon_model_set_service_state(daemon_service_state_t new_state)
{
	daemon_service_state_t old_state = daemon_model.daemon_service_state;

	if (old_state != new_state)
	{
		/* state changed */
		daemon_model.daemon_service_state = new_state;

		if (new_state == DAEMON_SERVICE_READY) {
			logger_log_debug("Daemon service reached state DAEMON_SERVICE_READY.");

			/* notify systemd that the service has been started successfully */
			sd_notify(0, "READY=1");
		}

		if (new_state == DAEMON_SERVICE_SHUTDOWN_PENDING) {
			logger_log_debug("Shutdown of daemon service is pending.");
		}
	}
}

daemon_level_state_t daemon_model_get_level_state(void)
{
	return daemon_model.daemon_level_state;
}

const char *daemon_model_get_level_state_name(daemon_level_state_t state)
{
	return daemon_level_state_names[state];
}

daemon_service_state_t daemon_model_get_service_state(void)
{
	return daemon_model.daemon_service_state;
}

const char *daemon_model_get_service_state_name(daemon_service_state_t state)
{
	return daemon_service_state_names[state];
}

static void daemon_model_set_target_level_for_change(security_level_t *target_level,
										security_level_t current_level,
										security_level_t new_target_level)
{
	if ((current_level == new_target_level) &&
	    (new_target_level != LEVEL_IS_INVALID))
		*target_level = TARGET_LEVEL_IS_SAME_AS_CURRENT;
	else
		*target_level = new_target_level;
}

void daemon_model_set_state_recovery_ongoing(security_level_t recovery_level)
{
	if (recovery_level > MAX_LEVEL) {
		logger_log_error("Invalid recovery level passed to daemon_model_set_state_recovery_ongoing - using fail-safe level instead");
		recovery_level = FAIL_SAFE_LEVEL;
	}

	daemon_model_set_target_level_for_change(&(daemon_model.targeted_active_level),
											 daemon_model.current_active_level,
											 recovery_level);

	daemon_model_set_target_level_for_change(&(daemon_model.persistent_model.targeted_persisted_level),
											 daemon_model.persistent_model.current_persisted_level,
											 recovery_level);

	daemon_model.daemon_level_state = DAEMON_LEVEL_RECOVERY_ONGOING;
}

void daemon_model_set_state_change_ongoing(security_level_t new_active_level, security_level_t new_persisted_level)
{
	if (new_active_level > MAX_LEVEL) {
		logger_log_error("Invalid new active level passed to daemon_model_set_state_change_ongoing - using fail-safe level instead");
		new_active_level = FAIL_SAFE_LEVEL;
	}

	if (new_persisted_level > MAX_LEVEL) {
		logger_log_error("Invalid new persistent level passed to daemon_model_set_state_change_ongoing - using fail-safe level instead");
		new_persisted_level = FAIL_SAFE_LEVEL;
	}

	daemon_model_set_target_level_for_change(&(daemon_model.targeted_active_level),
											 daemon_model.current_active_level,
											 new_active_level);

	daemon_model_set_target_level_for_change(&(daemon_model.persistent_model.targeted_persisted_level),
											 daemon_model.persistent_model.current_persisted_level,
											 new_persisted_level);

	daemon_model.daemon_level_state = DAEMON_LEVEL_CHANGE_ONGOING;
}

void daemon_model_set_stable_levels(security_level_t active_level,
				    security_level_t persisted_level,
				    bool completed_with_error)
{
	if (active_level > MAX_LEVEL) {
		logger_log_error("Invalid active level passed to daemon_model_set_states - mark invalid");
		active_level = LEVEL_IS_INVALID;
	}

	if (persisted_level > MAX_LEVEL) {
		logger_log_error("Invalid persistent level passed to daemon_model_set_states - mark invalid");
		persisted_level = LEVEL_IS_INVALID;
	}

	daemon_model.current_active_level=active_level;
	daemon_model.persistent_model.current_persisted_level=persisted_level;

	if (completed_with_error) {
		/* use special value for target to mark that the level change was
		 * completed with errors */
		daemon_model.targeted_active_level=LEVEL_CHANGE_COMPLETED_WITH_ERRORS;
		daemon_model.persistent_model.targeted_persisted_level=LEVEL_CHANGE_COMPLETED_WITH_ERRORS;
	} else {
		/* use same values as for current */
		daemon_model.targeted_active_level=active_level;
		daemon_model.persistent_model.targeted_persisted_level=persisted_level;
	}
}

void daemon_model_schedule_fail_safe_recovery()
{
	daemon_model.targeted_active_level=LEVEL_RECOVERY_TO_FAIL_SAFE;
	daemon_model.persistent_model.targeted_persisted_level=LEVEL_RECOVERY_TO_FAIL_SAFE;
}

void daemon_model_set_state_idle(bool recovery_on_next_startup)
{
	if (recovery_on_next_startup) {
		/* Change the Daemon state as we need a recovery on startup to heal the situation */
		daemon_model.daemon_level_state = DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED;
	} else {
		/* setting the level worked OK */
		daemon_model.daemon_level_state=DAEMON_LEVEL_STABLE;
	}

	/* the initialization of the service is done, when the
	 * level state reaches idle */
	switch (daemon_model.daemon_service_state) {
	case DAEMON_SERVICE_INITIALIZATION:
		daemon_model_set_service_state(DAEMON_SERVICE_WAITING_FOR_APP_IFACE_READY);
		break;
	case DAEMON_SERVICE_WAITING_FOR_LEVEL_CHANGE_RECOVERY_ON_STARTUP:
		daemon_model_set_service_state(DAEMON_SERVICE_READY);
		break;
	default:
		/* ignore */
		break;
	}
}

void daemon_model_set_state_shutting_down(void)
{
	daemon_model_set_service_state(DAEMON_SERVICE_SHUTDOWN_PENDING);
}

void daemon_model_app_iface_set_ready(void)
{
	switch (daemon_model.daemon_service_state) {
	case DAEMON_SERVICE_INITIALIZATION:
		daemon_model_set_service_state(DAEMON_SERVICE_WAITING_FOR_LEVEL_CHANGE_RECOVERY_ON_STARTUP);
		break;
	case DAEMON_SERVICE_WAITING_FOR_APP_IFACE_READY:
		daemon_model_set_service_state(DAEMON_SERVICE_READY);
		break;
	case DAEMON_SERVICE_SHUTDOWN_PENDING:
		/* ignore - this might happen if shutdown trigger happens before DBUS is ready */
		break;
	default:
		logger_log_error("DBUS ready triggered in unexpected service state (%s)",
				daemon_model_get_service_state_name(daemon_model.daemon_service_state));
	}
}
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- private function definition -------------------------------------------------
static void daemon_model_log_level(const char* logmsg, security_level_t level)
{
	const char *level_str = "";

	switch (level) {
	case LEVEL_CHANGE_COMPLETED_WITH_ERRORS:
		level_str=" (LEVEL_CHANGE_COMPLETED_WITH_ERRORS)";
		break;
	case LEVEL_RECOVERY_TO_FAIL_SAFE:
		level_str=" (LEVEL_RECOVERY_TO_FAIL_SAFE)";
		break;
	case TARGET_LEVEL_IS_SAME_AS_CURRENT:
		level_str=" (TARGET_LEVEL_IS_SAME_AS_CURRENT)";
		break;
	case LEVEL_IS_INVALID:
		level_str=" (LEVEL_IS_INVALID)";
		break;
	default:
		if (level > MAX_LEVEL)
			level_str=" (unknown)";
	}

	logger_log_debug("%s%d%s", logmsg, (int)level, level_str);
}

static error_code_t daemon_model_load_persistent_model(persistent_model_t *model)
{
	const char *fn;
	error_code_t result;

	fn=configuration_get_persisted_state_path();

    /* write initial values to peristent_model
     * In case loading the state file fails, these settings will trigger a recovery of the FAIL_SAFE_LEVEL
     */
	model->current_persisted_level = LEVEL_IS_INVALID;
	model->targeted_persisted_level = LEVEL_IS_INVALID;

	result=sdc_access_load_state_file(fn, model, sizeof(persistent_model_t));

	if (result==RESULT_WRAPPED_FILE_CORRUPT)
		logger_log_error("Unable to read persistent state from %s. Content of file is corrupt.",fn);
	else if (result==RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES)
		logger_log_error("Unable to read persistent state from %s. Message: %s",fn,strerror(errno));
	else if (result==RESULT_CRYPTOGRAPHIC_OPERATION_FAILED || result==RESULT_NORESOURCE)
		logger_log_error("Decryption of persistent ALD state failed.");
	else
	{
		//we are done, provide some info
		logger_log_debug("DAEMON_MODEL - Reading persistent state from file %s successfully.",fn);
		daemon_model_log_level("\t\tCurrent persisted level: ", model->current_persisted_level);
		daemon_model_log_level("\t\tTargeted persisted level: ",model->targeted_persisted_level);
	}

	return result;
}

static error_code_t daemon_model_store_persistent_model(persistent_model_t *model)
{
	const char *fn;
	error_code_t result;

	fn=configuration_get_persisted_state_path();
	result=sdc_access_store_state_file(fn, model, sizeof(persistent_model_t));

	if (result==RESULT_WRAPPED_FILE_CORRUPT || result==RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES)
		logger_log_error("Unable to write persistent state to %s. Message: %s",fn,strerror(errno));
	else if (result==RESULT_CRYPTOGRAPHIC_OPERATION_FAILED || result==RESULT_NORESOURCE)
		logger_log_error("Encryption of persistent ALD state failed.");
	else
	{
		//we are done, provide some info
		logger_log_debug("DAEMON_MODEL - Written persistent state to file %s successfully.",fn);
		daemon_model_log_level("\t\tCurrent persisted level: ", model->current_persisted_level);
		daemon_model_log_level("\t\tTargeted persisted level: ", model->targeted_persisted_level);
	}

	return RESULT_OK;
}
//--------------------------------------------------------------------------------------------------------------------
